/*************************************************************************
 * The contents of this file are subject to the MYRICOM MX AND GM-2      *
 * MAPPING SOFTWARE AND DOCUMENTATION LICENSE (the "License"); User may  *
 * not use this file except in compliance with the License.  The full    *
 * text of the License can found in mapper directory in LICENSE.TXT      *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

#include "lx.h"
#include "lx_routing.h"

void lx_routing_init_ports (int ports [LX_XBAR_SIZE], int start);
void lx_routing_reclaim_xbars (lx_map_t*m, lx_queue_t*seen, lx_queue_t*frontier);

static void lx_routing_mark_up (lx_map_t*m)
{
  int i;
  
  lx_node_t*n;
  insist (m);

  for (i = 0; i < m->num_xbars; i++)
  {
    mi_punt (m->lx);
    n = mi_index2node (m->xbar_array [i]);
    insist (n && n->xbar);
    insist (LX_UP && LX_DOWN);
    n->xbar = LX_UP;
  }
  except:;
}

void lx_routing_init_ports (int ports [LX_XBAR_SIZE], int start)
{
  int i;
  
  insist (ports);
  
  for (i = 0; i < LX_XBAR_SIZE; i++)
    ports [i] = (i + start) % LX_XBAR_SIZE;

  except:;  
}

void lx_routing_shuffle_ports (int ports [LX_XBAR_SIZE])
{
  int i, a, b, t;
  insist (ports);
  
  for (i = 0; i < LX_XBAR_SIZE; i++)
    ports [i] = i;

  for (i = 0; i < 32; i++)
  {
    a = mi_rand () % LX_XBAR_SIZE;
    b = mi_rand () % LX_XBAR_SIZE;
    insist (a >= 0 && a < LX_XBAR_SIZE);
    insist (b >= 0 && b < LX_XBAR_SIZE);
    
    t = ports [a];
    ports [a] = ports [b];
    ports [b] = t;
  }
  except:;
}

void lx_routing_sort_ports (lx_map_t*m, int ports [LX_XBAR_SIZE], lx_node_t*n)
{
  int i, j, t, lowest_port;
  mi_weight_t lowest_weight = 0;
  
  insist (m && ports && n && n->xbar);

  for (i = 0; i < LX_XBAR_SIZE; i++)
  {
    lowest_port = -1;
    mi_punt (m->lx);

    for (j = i; j < LX_XBAR_SIZE; j++)
    {
      insist (ports [j] >= 0 && ports [j] < LX_XBAR_SIZE);
      if (lowest_port < 0 || n->links [ports [j]].weight < lowest_weight)
      {
	lowest_port = j;
	lowest_weight = n->links [ports [j]].weight;
      }
    }
    insist (lowest_port >= 0 && lowest_port < LX_XBAR_SIZE);

    t = ports [i];
    ports [i] = ports [lowest_port];
    ports [lowest_port] = t;
  
  }
  
  except:;
}

int lx_routing_shortest_path (lx_map_t*m, int iport, lx_node_t*me, int updown)
{
  int i, j, k, last_port, direction, oiport, total;
  lx_queue_t frontier, seen;
  lx_queue_t sought;
  lx_node_t*n, *o;
  int ports [LX_XBAR_SIZE];

  insist (m && me);
  insist (iport >=0 && iport < MI_MY_SIZE);
 
  lx_bzero ((char*) &frontier, sizeof (frontier));
  lx_bzero ((char*) &seen, sizeof (seen));
  lx_bzero ((char*) &sought, sizeof (sought));

  n = lx_get_node (me, iport);

  if (!n || !n->xbar)
  {
    for (j = 0; j < LX_HOST_SIZE; j++)
    {
      if (n) 
	lx_host_c (n)->routes [iport] [j].length = 0;
      lx_host_c (me)->routes [iport] [j].length = 0;
    }
    return 1;
  }  

  if (updown)
    lx_routing_mark_up (m);  
  
  lx_put (&frontier, lx_remove_from_anywhere (n));
  n->original.port = lx_get_port (me, iport) - n->first;
  lx_xbar_c (n)->routes [iport].length = 0;
    
  for (i = 0; i < m->num_hosts; i++)
  {
    n = mi_index2node (m->host_array [i]);
    insist (n && !n->xbar);

    for (j = 0; j < LX_HOST_SIZE; j++)
    {
      lx_host_c (n)->num_routes [j] = lx_get_node (n, j) ? 1 : 0;
      lx_host_c (n)->routes [iport] [j].length = 0;
    }

    insist (lx_queued (&m->hosts, n));
    lx_put (&sought, lx_remove_from_anywhere (n));
    mi_punt (m->lx);
  }
  
  while ((n = lx_get (&frontier)))
  {
    insist (n->xbar);
    lx_put (&seen, n);

    lx_routing_shuffle_ports (ports);
    lx_routing_sort_ports (m, ports, n);
    
    mi_punt (m->lx);

    /*printf ("exploring " lx_node_format "\n", lx_node_args (n));*/

    for (j = 0; j < LX_XBAR_SIZE; j++)
    {
      i = ports [j];
      mi_punt (m->lx);
    
      if (!(o = lx_get_node (n, n->first + i)))
	continue;

      insist (o != n);
      
      if (o->xbar)
      {
	if (!lx_queued (&m->xbars, o))
	  continue;
	
	if (!(direction = lx_xbar_c (n)->level - lx_xbar_c (o)->level))
	  direction = n->index - o->index;

	insist (direction);
	insist (n->xbar == LX_UP || n->xbar == LX_DOWN);
	
	if (updown && !(n->xbar == LX_UP || (n->xbar == LX_DOWN && direction > 0)))
	  continue;

	o->xbar = direction < 0 ? LX_UP : LX_DOWN;
	
	/*mi_c (("added " lx_node_format " to frontier", lx_node_args (o)));*/

	lx_append_route (&lx_xbar_c (o)->routes [iport], &lx_xbar_c (n)->routes [iport], i - n->original.port);
	o->original.port = lx_get_port (n, n->first + i) - o->first;
	lx_put (&frontier, lx_remove_from_anywhere (o));
      }
      else if (lx_queued (&sought, o))
      {
	oiport = lx_get_port (n, n->first + i) - o->first;
	insist (oiport >= 0 && oiport < LX_HOST_SIZE);
	insist (lx_host_c (o)->num_routes [oiport] == 1);

	/*mi_c (("found " lx_node_format " oiport %d", lx_node_args (o), oiport));*/
	
	lx_append_route (&lx_host_c (o)->routes [iport] [oiport], &lx_xbar_c (n)->routes [iport], i - n->original.port);

	insist (lx_follow (me, iport, &lx_host_c (o)->routes [iport] [oiport], &last_port) == o && last_port == oiport);
	
	lx_host_c (o)->num_routes [oiport] = 0;
	for (k = total = 0; k < LX_HOST_SIZE; k++)
	  total += lx_host_c (o)->num_routes [k];

	if (!total)
	  lx_put (&m->hosts, lx_remove_from_anywhere (o));
      }
    }
  }

  insist (!sought.count);
  insist (seen.count <= m->num_xbars);
  
  if (!(k = (seen.count == m->num_xbars)))
  {
    mi_c (("there were %d unreachable xbars", m->num_xbars - seen.count));
  }
  
  while ((n = lx_get (&seen)))
  {
    lx_put (&m->xbars, lx_remove_from_anywhere (n));
    mi_punt (m->lx);
  }

  return k;
  except: return 0;
}

int lx_routing_number_xbars_clos (lx_map_t*m, int*num_spines)
{
  lx_queue_t frontier, numbered;
  lx_node_t*n, *o;
  int max_ports, level, highest, i;
  int not_clos = 0;
  
  insist (m && num_spines);
  
  lx_bzero ((char*) &frontier, sizeof (frontier));
  lx_bzero ((char*) &numbered, sizeof (numbered));

  *num_spines = 0;
  
  while ((n = lx_get (&m->hosts)))
  {
    insist (!n->xbar);
    mi_punt (m->lx);
    lx_put (&frontier, n);
  }
  
  highest = 0;
  
  while ((n = lx_get (&frontier)))
  {
    max_ports = n->xbar ? LX_XBAR_SIZE : LX_HOST_SIZE;
    for (i = 0; i < max_ports; i++)
    {
      mi_punt (m->lx);

      if ((o = lx_get_node (n, i + n->first)))
      {
	if (lx_queued (&m->xbars, o))
	{
	  insist (o != n);
	  insist (o->xbar);
	
	  if ((level = (n->xbar ? lx_xbar_c (n)->level : 0) + 1) > highest)
	  {
	    *num_spines = 0;
	    highest = level;
	  }
	  if (level == highest)
	    (*num_spines)++;
	  lx_xbar_c (o)->level = level;
	  lx_put (&frontier, lx_remove_from_anywhere (o));
	}
      }
    }
    lx_put (n->xbar ? &numbered : &m->hosts, n);
  }
   
  not_clos = numbered.count != m->num_xbars;

  while ((n = lx_get (&numbered)))
  {
    lx_put (&m->xbars, n);
    mi_punt (m->lx);
  }

  return not_clos ? 0 : highest;
  except:
  return 0;
}


void lx_routing_reclaim_xbars (lx_map_t*m, lx_queue_t*seen, lx_queue_t*frontier)
{
  lx_node_t*n;
  insist (m && seen && frontier);
      
  while ((n = lx_get (seen)))
  {
    lx_put (&m->xbars, lx_remove_from_anywhere (n));
    mi_punt (m->lx);
  }
  while ((n = lx_get (frontier)))
  {
    lx_put (&m->xbars, lx_remove_from_anywhere (n));
    mi_punt (m->lx);
  }
  insist (m->xbars.count == m->num_xbars);
  except:;
}


void lx_routing_forget (lx_map_t*m)
{
  int i, j;
  
  lx_node_t*n;
  insist (m);

  for (i = 0; i < m->num_xbars; i++)
  {
    mi_punt (m->lx);
    n = mi_index2node (m->xbar_array [i]);
    insist (n && n->xbar);
    lx_xbar_c (n)->last = 0;
    for (j = 0; j < LX_XBAR_SIZE; j++)
      n->links [j].weight = 0;
  }
  except:;
}

static int lx_routing_depth_first (lx_map_t*m, lx_node_t*me, int iport, lx_node_t*n, lx_queue_t*sought, lx_queue_t*path, int remaining, int seek_size, int order, int up, int depth, int max_depth, int shortest)
{
  int r, i, j, k, last_port, direction, oiport, total, sought_before;
  lx_node_t*o;
  int ports [LX_XBAR_SIZE];
  
  insist (m && me && n && sought && n->xbar);
  insist (iport >=0 && iport < MI_MY_SIZE);

#ifndef MI_XM
  lx_wait (m->lx, MI_POLL_USECS, 0, 1);
#endif

  if (!sought->count || !remaining || remaining - sought->count >= seek_size)
    return 0;
  
  if (max_depth && depth > max_depth)
    return 1;
  
  insist (lx_queued (&m->xbars, n));
  insist (path->count == depth);
  lx_put (path, lx_remove_from_anywhere (n));

  /*
    for (k=0;k<depth;k++)printf(" ");
    printf ("exploring " lx_node_format " depth %d\n", lx_node_args (n), depth);
    insist (n->routes [iport].length == depth);
  */
  
  switch (order)
  {
    case LX_ROUTING_RANDOM:
      lx_routing_init_ports (ports, 0);
      lx_routing_shuffle_ports (ports);
      break;
    case LX_ROUTING_LEAST:
      lx_routing_init_ports (ports, 0);
      lx_routing_shuffle_ports (ports);
      lx_routing_sort_ports (m, ports, n);
      break;
    case LX_ROUTING_NEXT:
      lx_routing_init_ports (ports, lx_xbar_c (n)->last++);
      break;
    default:
      insist (0);
  }
  mi_punt (m->lx);

  for (j = 0; j < LX_XBAR_SIZE; j++)
  {
    i = ports [j];
    mi_punt (m->lx);
   
    if ((o = lx_get_node (n, i + n->first)) && !lx_queued (path, o) && o != n)
    {  
      insist (o != n);
      
      if (o->xbar)
      {
	if (!(direction = lx_xbar_c (n)->level - lx_xbar_c (o)->level))
	  direction = n->index - o->index;

	insist (direction);
	
	if (up || (!up && direction > 0))
	{

	  /* check skip depths for this xbar */
	  if (direction < 0)
	  {
	    if (lx_xbar_c (o)->up_skip_depth <= depth) continue;
	  }
	  else
	  {
	    if (lx_xbar_c (o)->down_skip_depth <= depth) continue;
	  }

	  /*
	    for (k=0;k<depth;k++)printf(" ");
	    printf ("found " lx_node_format " level is %d, depth is %d\n", lx_node_args (o), lx_xbar_c (o)->level, depth);
	  */

	  lx_append_route (&lx_xbar_c (o)->routes [iport], &lx_xbar_c (n)->routes [iport], i - n->original.port);

	  /*
	    for (k=0;k<depth;k++)printf(" ");
	    printf ("route is %s\n", lx_print_route (o->routes [iport].hops, o->routes [iport].length));
	  */

	  o->original.port = lx_get_port (n, n->first + i) - o->first;
	  
	  sought_before = sought->count;
	  
	  r = lx_routing_depth_first (m, me, iport, o, sought, path, remaining, seek_size, order, direction < 0, depth + 1, max_depth, shortest);

	  insist (sought_before - sought->count >= 0);
	  
	  n->links [i].weight += sought_before - sought->count;
	  /*mi_c (("weight += %d", sought_before - sought->count));*/
	  
	  if (sought_before == sought->count)
	  {
	    if (direction < 0)
	    {
	      lx_xbar_c (o)->up_skip_depth = depth;
	    }
	    else
	    {
	      lx_xbar_c (o)->down_skip_depth = depth;
	    }
	  }
	  if (!r)
	  {
	    lx_put (&m->xbars, lx_remove_from_anywhere (n));
	    return 0;
	  }
	}
      }
      else if (lx_queued (sought, o))
      {
	oiport = lx_get_port (n, n->first + i) - o->first;
	insist (oiport >= 0 && oiport < LX_HOST_SIZE);

	if (lx_host_c (o)->num_routes [oiport] != 1) 
	  continue;

	if (shortest && depth >= o->original.port)
	  continue;
	
	/*mi_c (("found " lx_node_format " oiport %d", lx_node_args (o), oiport));*/

	insist (lx_host_c (o)->num_routes [oiport] == 1);

	lx_append_route (&lx_host_c (o)->routes [iport] [oiport], &lx_xbar_c (n)->routes [iport], i - n->original.port);

	/*mi_c (("route is %s", lx_print_route (lx_host_c (o)->routes [iport] [oiport].hops, lx_host_c (o)->routes [iport] [oiport].length)));*/
	mi_punt (m->lx);

	insist (lx_follow (me, iport, &lx_host_c (o)->routes [iport] [oiport], &last_port) == o && oiport == last_port);

	lx_host_c (o)->num_routes [oiport] = 0;
	for (k = total = 0; k < LX_HOST_SIZE; k++)
	  total += lx_host_c (o)->num_routes [k];

	if (!total)
	{
	  lx_put (&m->hosts, lx_remove_from_anywhere (o));
	  insist (remaining - sought->count >= 0 && remaining - sought->count <= m->num_hosts);
	
	  if (remaining - sought->count >= seek_size)
	  {
	    lx_put (&m->xbars, lx_remove_from_anywhere (n));
	    return 0;
	  }
	}
      }
    }
  }
  lx_put (&m->xbars, lx_remove_from_anywhere (n));
  return 1;
  except: return 0;
}

int d_not_clos = 0;

int lx_routing_careful_routes (lx_map_t*m, int iport, lx_node_t*me, int seek_size, int order, int shortest)
{
  int i, j, remaining, longest;
  lx_queue_t sought, path;
  lx_node_t*n;
  
  insist (m && me);
  
  if (seek_size <= 0)
    seek_size = m->num_hosts;

  n = lx_get_node (me, iport);

  if (!n || !n->xbar)
  {
    for (j = 0; j < LX_HOST_SIZE; j++)
    {
      if (n) 
	lx_host_c (n)->routes [iport] [j].length = 0;
      lx_host_c (me)->routes [iport] [j].length = 0;
    }
    return 1;
  }

  lx_bzero ((char*) &sought, sizeof (sought));
  lx_bzero ((char*) &path, sizeof (path));
  
  longest = 0;
  
  /* Reset all skip depths */
  for (i = 0; i < m->num_xbars; i++)
  {
    n = mi_index2node (m->xbar_array [i]);
    lx_xbar_c (n)->down_skip_depth = 99;
    lx_xbar_c (n)->up_skip_depth = 99;
  }

  for (i = 0; i < m->num_hosts; i++)
  {
    n = mi_index2node (m->host_array [i]);
    insist (n && !n->xbar);

    for (j = 0; j < LX_HOST_SIZE; j++)
    {
      lx_host_c (n)->num_routes [j] = lx_get_node (n, j) ? 1 : 0;
      lx_host_c (n)->routes [iport] [j].length = 0;
    }
  
    if (n->original.port > longest)
      longest = n->original.port;
    
    insist (lx_queued (&m->hosts, n));
    lx_put (&sought, lx_remove_from_anywhere (n));
    mi_punt (m->lx);
  }

  while ((remaining = sought.count))
  {

#ifndef MI_XM
    lx_wait (m->lx, MI_POLL_USECS, 0, 1);
#endif

    n = lx_get_node (me, iport);
    insist (n && n->xbar);
    n->original.port = lx_get_port (me, iport) - n->first;
    lx_xbar_c (n)->routes [iport].length = 0;

    /*mi_c (("starting at " lx_node_format " level is %d", lx_node_args (n), lx_xbar_c (n)->level));*/

    lx_routing_depth_first (m, me, iport, n, &sought, &path, remaining, seek_size, order, 1, 0, longest - 1, shortest);

    insist (path.count == 0);

    if (sought.count == remaining)
    {
      mi_c (("not a clos"));
      d_not_clos++;
      while ((n = lx_get (&sought)))
      {
	lx_put (&m->hosts, n);
	mi_punt (m->lx);
      }      
      return 0;
    }
  }
 
  return 1;
  except: return 0;
}
